/*
 * ALSA driver for Panasonic UniPhier series.
 * 
 * Copyright (c) 2013 Panasonic corporation.
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; version 2
 * of the License.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, see <http://www.gnu.org/licenses/>.
 */

#include <linux/module.h>
#include <linux/moduleparam.h>
#include <linux/init.h>

#include <linux/kernel.h>
#include <linux/errno.h>
#include <linux/types.h>
#include <linux/spinlock.h>
#include <linux/vmalloc.h>
#include <linux/kthread.h>
#include <linux/delay.h>
#include <linux/cdev.h>
#include <linux/dma-mapping.h>

#include "mn2ws-pcm.h"

MODULE_AUTHOR("Katsuhiro Suzuki <suzuki.katsuhiro002@jp.panasonic.com>");
MODULE_DESCRIPTION("Panasonic UniPhier PCM Interface Common Driver");
MODULE_LICENSE("GPL");

const char *mn2ws_pcm_pcmif_get_type(struct mn2ws_pcm_pcmif *pcmif)
{
	const char *name = "?";
	
	switch (pcmif->type) {
	case MN2WS_PCMIF_PLAYBACK:
		name = "p";
		break;
	case MN2WS_PCMIF_CAPTURE:
		name = "c";
		break;
	default:
		name = "u";
		break;
	}
	
	return name;
}

int mn2ws_pcm_pcmif_init(struct mn2ws_pcm_dev *d, struct mn2ws_pcm_pcmif *pcmif)
{
	int result;
	
	DPRINTF("%s\n", __func__);
	
	if (!pcmif->desc->enabled) {
		DPRINTF("ch%d_%s: disabled.\n", 
			d->ch, mn2ws_pcm_pcmif_get_type(pcmif));
		return 0;
	}
	
	spin_lock_init(&pcmif->spin);
	init_waitqueue_head(&pcmif->q);
	
	pcmif->start_trigger = 0;
	pcmif->stop_trigger = 0;
	pcmif->silent_mode = 1;
	
	//check the necessary callbacks
	if (!pcmif->desc->hardware
		|| !pcmif->desc->setup
		|| !pcmif->desc->start
		|| !pcmif->desc->stop
		|| !pcmif->desc->get_hwptr
		|| !pcmif->desc->wait_hwevent
		|| !pcmif->desc->copy
		|| !pcmif->desc->silence) {
		PRINTF_WARN("ch%d_%s: BUG! one of methods is NULL.\n", 
			d->ch, mn2ws_pcm_pcmif_get_type(pcmif));
		snd_BUG();
	}
	
	//allocate the HW buffer
	result = mn2ws_pcm_pcmif_alloc_hw_buffer(d, pcmif);
	if (result != 0) {
		PRINTF_WARN("ch%d_%s: failed to allocate HW buffer.\n", 
			d->ch, mn2ws_pcm_pcmif_get_type(pcmif));
		return result;
	}
	
	//create transfer thread
	pcmif->thread = kthread_run(pcmif->thread_main, d, 
		"ch%d%s_%s", d->ch, mn2ws_pcm_pcmif_get_type(pcmif), 
		get_pcm_dev_name());
	if (pcmif->thread == NULL) {
		PRINTF_WARN("ch%d_%s: failed to kthread_create.\n", 
			d->ch, mn2ws_pcm_pcmif_get_type(pcmif));
		return -EFAULT;
	}
	
	return 0;
}

int mn2ws_pcm_pcmif_exit(struct mn2ws_pcm_dev *d, struct mn2ws_pcm_pcmif *pcmif)
{
	DPRINTF("%s\n", __func__);
	
	if (pcmif->thread) {
		kthread_stop(pcmif->thread);
		pcmif->thread = NULL;
	}
	
	mn2ws_pcm_pcmif_free_hw_buffer(d, pcmif);
	
	return 0;
}

int mn2ws_pcm_pcmif_alloc_hw_buffer(struct mn2ws_pcm_dev *d, struct mn2ws_pcm_pcmif *pcmif)
{
	DPRINTF("%s\n", __func__);
	
	if (pcmif->desc->phys_addr_buf == 0) {
		//allocate buffer by kernel
		pcmif->buf = dma_alloc_coherent(NULL, 
			max((int)PAGE_SIZE, (int)pcmif->desc->size_buf), 
			&pcmif->paddr_buf, GFP_KERNEL);
	} else {
		//use fixed buffer
		pcmif->paddr_buf = pcmif->desc->phys_addr_buf;
		
		//map the HW buffer in uncached mode
		pcmif->buf = ioremap_nocache(pcmif->paddr_buf, 
			pcmif->desc->size_buf);
	}
	if (pcmif->buf == 0) {
		PRINTF_WARN("ch%d_%s: failed to alloc.\n", 
			d->ch, mn2ws_pcm_pcmif_get_type(pcmif));
		return -EFAULT;
	}
	memset(pcmif->buf, 0, pcmif->desc->size_buf);
	
	PRINTF_NOTE("ch%d_%s: paddr:0x%08x, map:0x%p, size:%d\n", 
		d->ch, mn2ws_pcm_pcmif_get_type(pcmif), 
		pcmif->paddr_buf, pcmif->buf, pcmif->desc->size_buf);
	
	return 0;
}

int mn2ws_pcm_pcmif_free_hw_buffer(struct mn2ws_pcm_dev *d, struct mn2ws_pcm_pcmif *pcmif)
{
	DPRINTF("%s\n", __func__);
	
	if (pcmif->buf) {
		if (pcmif->desc->phys_addr_buf == 0) {
			//allocate buffer by kernel
			dma_free_coherent(NULL, 
				max((int)PAGE_SIZE, 
					(int)pcmif->desc->size_buf), 
				pcmif->buf, pcmif->paddr_buf);
		} else {
			//use fixed buffer
			iounmap(pcmif->buf);
		}
		pcmif->buf = NULL;
	}
	
	return 0;
}

int mn2ws_pcm_pcmif_alloc_bounce_buffer(struct snd_pcm_substream *substream, size_t size, struct mn2ws_pcm_pcmif *pcmif)
{
	struct snd_pcm_runtime *runtime = substream->runtime;
	struct mn2ws_pcm_chip *chip = snd_pcm_substream_chip(substream);
	struct mn2ws_pcm_dev *d = chip->device;
	size_t dma_size;
	
	DPRINTF("%s\n", __func__);
	
	if (runtime->dma_area) {
		dma_size = runtime->dma_bytes;
		
		vfree(runtime->dma_area);
		runtime->dma_area = NULL;
		runtime->dma_bytes = 0;
		
		DPRINTF("ch%d_%s: realloc bounce dma_area %d[bytes].\n", 
			d->ch, mn2ws_pcm_pcmif_get_type(pcmif), (int)dma_size);
	}
	
	dma_size = size;
	
	runtime->dma_area = vmalloc(dma_size);
	if (runtime->dma_area == NULL) {
		PRINTF_WARN("ch%d_%s: cannot allocate bounce dma_area.\n", 
			d->ch, mn2ws_pcm_pcmif_get_type(pcmif));
		return -ENOMEM;
	}
	runtime->dma_bytes = dma_size;
	
	DPRINTF("ch%d_%s: require bounce dma_area %d[bytes].\n", 
		d->ch, mn2ws_pcm_pcmif_get_type(pcmif), (int)dma_size);
	
	return 0;
}

int mn2ws_pcm_pcmif_free_bounce_buffer(struct snd_pcm_substream *substream, struct mn2ws_pcm_pcmif *pcmif)
{
	struct snd_pcm_runtime *runtime = substream->runtime;
	struct mn2ws_pcm_chip *chip = snd_pcm_substream_chip(substream);
	struct mn2ws_pcm_dev *d = chip->device;
	size_t dma_size;
	
	DPRINTF("%s\n", __func__);
	
	if (runtime->dma_area) {
		dma_size = runtime->dma_bytes;

		vfree(runtime->dma_area);
		runtime->dma_area = NULL;
		runtime->dma_bytes = 0;
		
		DPRINTF("ch%d_%s: free ALSA dma_area %d[bytes].\n", 
			d->ch, mn2ws_pcm_pcmif_get_type(pcmif), (int)dma_size);
	}
	
	return 0;
}

int mn2ws_pcm_pcmif_reset_hw_ringbuf(struct snd_pcm_substream *substream, struct mn2ws_pcm_pcmif *pcmif)
{
	struct mn2ws_pcm_chip *chip = snd_pcm_substream_chip(substream);
	struct mn2ws_pcm_dev *d = chip->device;
	
	DPRINTF("%s\n", __func__);
	
	if (pcmif->desc->size_use_fix == 0) {
		//Use default size
		pcmif->size_buf_use = pcmif->desc->size_buf;
	} else {
		//Use HW constraint size
		pcmif->size_buf_use = pcmif->desc->size_use_fix;
	}
	if (pcmif->size_buf_use > pcmif->desc->size_buf) {
		PRINTF_WARN("ch%d_%s: BUG!! HW buffer is exceeded maximum size"
			"(size_buf_use:0x%08x, size_buf:0x%08x)\n", 
			d->ch, mn2ws_pcm_pcmif_get_type(pcmif), 
			pcmif->size_buf_use, pcmif->desc->size_buf);
		return -ENOMEM;
	}
	
	init_ringbuf(&pcmif->hw, pcmif->buf, pcmif->size_buf_use);
	fill_ringbuf(&pcmif->hw, 0);
	
	DPRINTF("ch%d_%s: HW ring buffer size:%d[bytes]\n", 
		d->ch, mn2ws_pcm_pcmif_get_type(pcmif), pcmif->size_buf_use);
	
	return 0;
}

int mn2ws_pcm_pcmif_reset_bounce_ringbuf(struct snd_pcm_substream *substream, struct mn2ws_pcm_pcmif *pcmif)
{
	struct snd_pcm_runtime *runtime = substream->runtime;
	struct mn2ws_pcm_chip *chip = snd_pcm_substream_chip(substream);
	struct mn2ws_pcm_dev *d = chip->device;
	
	DPRINTF("%s\n", __func__);
	
	init_ringbuf(&pcmif->alsa, runtime->dma_area, runtime->dma_bytes);
	
	DPRINTF("ch%d_%s: bounce ring buffer size:%d[bytes]\n", 
		d->ch, mn2ws_pcm_pcmif_get_type(pcmif), runtime->dma_bytes);
	
	return 0;
}


/**
 * ϡɥκǽϤ 16kHz, 16bit, Mono ꤹ롣
 * 
 * PCMMIX ѡ
 * 
 * @param d PCM ǥХ
 * @param h ϡɥκǽ
 * @return ʤ 0Ԥʤ 0 ʳΥ顼
 */
int mn2ws_pcm_pcmif_hardware_16k_16b_1ch(struct mn2ws_pcm_dev *dev, struct snd_pcm_hardware *h)
{
	DPRINTF("%s\n", __func__);
	
	h->info = (SNDRV_PCM_INFO_MMAP |
		SNDRV_PCM_INFO_MMAP_VALID |
		SNDRV_PCM_INFO_BATCH |
		SNDRV_PCM_INFO_INTERLEAVED |
		//SNDRV_PCM_INFO_NONINTERLEAVED |
		SNDRV_PCM_INFO_BLOCK_TRANSFER); // | 
		//SNDRV_PCM_INFO_PAUSE);
	h->formats = SNDRV_PCM_FMTBIT_S16_BE;
	h->rates = SNDRV_PCM_RATE_16000;
	h->rate_min = 16000;
	h->rate_max = 16000;
	h->channels_min = 1;
	h->channels_max = 1;
	h->buffer_bytes_max = 1024 * 1024;
	h->period_bytes_min = 256;
	h->period_bytes_max = 256 * 1024;
	h->periods_min = 2;
	h->periods_max = 1024;
	h->fifo_size = 0;
	
	return 0;
}

/**
 * ϡɥκǽϤ 48kHz, 16bit, Stereo ꤹ롣
 * 
 * EXTPCM ѡ
 * 
 * @param d PCM ǥХ
 * @param h ϡɥκǽ
 * @return ʤ 0Ԥʤ 0 ʳΥ顼
 */
int mn2ws_pcm_pcmif_hardware_48k_16b_2ch(struct mn2ws_pcm_dev *dev, struct snd_pcm_hardware *h)
{
	DPRINTF("%s\n", __func__);
	
	h->info = (SNDRV_PCM_INFO_MMAP |
		SNDRV_PCM_INFO_MMAP_VALID |
		SNDRV_PCM_INFO_BATCH |
		SNDRV_PCM_INFO_INTERLEAVED |
		//SNDRV_PCM_INFO_NONINTERLEAVED |
		SNDRV_PCM_INFO_BLOCK_TRANSFER); // | 
		//SNDRV_PCM_INFO_PAUSE);
	h->formats = SNDRV_PCM_FMTBIT_S16_BE;
	h->rates = SNDRV_PCM_RATE_48000;
	h->rate_min = 48000;
	h->rate_max = 48000;
	h->channels_min = 2;
	h->channels_max = 2;
	h->buffer_bytes_max = 1024 * 1024;
	h->period_bytes_min = 256;
	h->period_bytes_max = 256 * 1024;
	h->periods_min = 2;
	h->periods_max = 1024;
	h->fifo_size = 0;
	
	return 0;
}

/**
 * ϡɥκǽϤ 48kHz, 32bit, Stereo ꤹ롣
 * 
 * 롼ץХåϿѡ
 * 
 * @param d PCM ǥХ
 * @param h ϡɥκǽ
 * @return ʤ 0Ԥʤ 0 ʳΥ顼
 */
int mn2ws_pcm_pcmif_hardware_48k_32b_2ch(struct mn2ws_pcm_dev *d, struct snd_pcm_hardware *h)
{
	DPRINTF("%s\n", __func__);
	
	h->info = (SNDRV_PCM_INFO_MMAP |
		SNDRV_PCM_INFO_MMAP_VALID |
		//SNDRV_PCM_INFO_BATCH |
		SNDRV_PCM_INFO_INTERLEAVED |
		//SNDRV_PCM_INFO_NONINTERLEAVED |
		SNDRV_PCM_INFO_BLOCK_TRANSFER); // | 
		//SNDRV_PCM_INFO_PAUSE);
	h->formats = SNDRV_PCM_FMTBIT_S32_BE;
	h->rates = SNDRV_PCM_RATE_48000;
	h->rate_min = 48000;
	h->rate_max = 48000;
	h->channels_min = 2;
	h->channels_max = 2;
	h->buffer_bytes_max = 1024 * 1024;
	h->period_bytes_min = 256;
	h->period_bytes_max = 256 * 1024;
	h->periods_min = 2;
	h->periods_max = 1024;
	h->fifo_size = 0;
	
	return 0;
}

/**
 * 14ms δ֡꡼פ롣
 * 
 * @param d PCM ǥХ
 * @return ʤ 0Ԥʤ 0 ʳΥ顼
 */
int mn2ws_pcm_pcmif_hrtimeout(struct mn2ws_pcm_dev *d)
{
	ktime_t expires;
	
	expires = ktime_add_ns(ktime_get(), 1000000);
	set_current_state(TASK_INTERRUPTIBLE);
	schedule_hrtimeout_range(&expires, 3000000, HRTIMER_MODE_ABS);
	
	return 0;
}

/**
 * HW Хåե˲ǡ򥳥ԡ롣
 * 
 * @param d PCM ǥХ
 * @param r ԡΥ󥰥Хåե
 * @param s ԡ륵
 * @return ʤ 0Ԥʤ 0 ʳΥ顼
 */
ssize_t mn2ws_pcm_play_copy_to_hw(struct mn2ws_pcm_dev *d, struct ringbuf *r, size_t s)
{
	ssize_t n;
	
	n = copy_ringbuf(&d->play.hw, r, s);
	
	return n;
}

/**
 * HW Хåե̵ǡ򥳥ԡ롣
 * 
 * @param d PCM ǥХ
 * @param s ԡ륵
 * @return ʤ 0Ԥʤ 0 ʳΥ顼
 */
ssize_t mn2ws_pcm_play_silence_to_hw(struct mn2ws_pcm_dev *d, size_t s)
{
	ssize_t n;
	
	n = write_silent_ringbuf(&d->play.hw, s);
	
	return n;
}

/**
 * ѥХåե˲ǡ򥳥ԡ롣
 * 
 * @param d PCM ǥХ
 * @param r ԡΥ󥰥Хåե
 * @param s ԡ륵
 * @return ʤ 0Ԥʤ 0 ʳΥ顼
 */
ssize_t mn2ws_pcm_cap_copy_to_alsa(struct mn2ws_pcm_dev *d, struct ringbuf *r, size_t s)
{
	ssize_t n;
	
	n = copy_ringbuf(&d->cap.alsa, r, s);
	
	return n;
}

/**
 * ѥХåե̵ǡ򥳥ԡ롣
 * 
 * @param d PCM ǥХ
 * @param s ԡ륵
 * @return ʤ 0Ԥʤ 0 ʳΥ顼
 */
ssize_t mn2ws_pcm_cap_silence_to_alsa(struct mn2ws_pcm_dev *d, size_t s)
{
	ssize_t n;
	
	n = write_silent_ringbuf(&d->cap.alsa, s);
	
	return n;
}




void mn2ws_pcm_dbg_pcmif_runtime(struct snd_pcm_substream *substream, struct mn2ws_pcm_pcmif *pcmif)
{
	struct snd_pcm_runtime *runtime = substream->runtime;
	struct mn2ws_pcm_chip *chip = snd_pcm_substream_chip(substream);
	struct mn2ws_pcm_dev *d = chip->device;
	size_t frame_bytes;

	frame_bytes = (runtime->frame_bits) >> 3;
	
	DPRINTF("ch%d_%s: runtime\n"
		"  rate       : %d\n"
		"  channels   : %d\n"
		"  period_size: %d\n"
		"  periods    : %d\n"
		"  buffer_size: %d\n"
		"  min_align  : %d\n"
		"  byte_align : %d\n"
		"  frame_bits : %d\n"
		"  sample_bits: %d\n"
		"  info       : %d\n"
		"  rate_num   : %d\n"
		"  rate_den   : %d\n"
		"", 
		d->ch, mn2ws_pcm_pcmif_get_type(pcmif), 
		(int)runtime->rate, 
		(int)runtime->channels, 
		(int)runtime->period_size, 
		(int)runtime->periods, 
		(int)runtime->buffer_size, 
		(int)runtime->min_align, 
		(int)runtime->byte_align, 
		(int)runtime->frame_bits, 
		(int)runtime->sample_bits, 
		(int)runtime->info, 
		(int)runtime->rate_num, 
		(int)runtime->rate_den);
	DPRINTF("ch%d_%s: frame_bytes: %d[bytes]\n", 
		d->ch, mn2ws_pcm_pcmif_get_type(pcmif), 
		frame_bytes);
}

void mn2ws_pcm_dbg_pcmif_hwbuf(struct mn2ws_pcm_dev *d, struct mn2ws_pcm_pcmif *pcmif)
{
	DPRINTF("ch%d_%s: hw:0x%p-0x%p(%d)\n", 
		d->ch, mn2ws_pcm_pcmif_get_type(pcmif), 
		pcmif->hw.start, pcmif->hw.start + pcmif->hw.len, 
		pcmif->hw.len);
}

void mn2ws_pcm_dbg_pcmif_ringbuf(struct mn2ws_pcm_dev *d, struct mn2ws_pcm_pcmif *pcmif)
{
	DPRINTF("ch%d_%s: hw(r:%05x,w:%05x,rem:%05x) "
		"alsa(r:%05x,w:%05x,rem:%05x)\n", 
		d->ch, mn2ws_pcm_pcmif_get_type(pcmif), 
		(int)get_rp_ringbuf(&pcmif->hw), 
		(int)get_wp_ringbuf(&pcmif->hw), 
		(int)get_remain_ringbuf(&pcmif->hw), 
		(int)get_rp_ringbuf(&pcmif->alsa), 
		(int)get_wp_ringbuf(&pcmif->alsa), 
		(int)get_remain_ringbuf(&pcmif->alsa));
}

void mn2ws_pcm_dbg_pcmif_ringbuf_maxmin(struct mn2ws_pcm_dev *d, struct mn2ws_pcm_pcmif *pcmif)
{
	static unsigned long j = 0;
	static size_t hw_max[8] = {0, }, hw_min[8] = {-1, };
	static size_t alsa_max[8] = {0, }, alsa_min[8] = {-1, };
	
	hw_max[d->ch] = max(hw_max[d->ch], 
		get_remain_ringbuf(&pcmif->hw));
	hw_min[d->ch] = min(hw_min[d->ch], 
		get_remain_ringbuf(&pcmif->hw));
	alsa_max[d->ch] = max(alsa_max[d->ch], 
		get_remain_ringbuf(&pcmif->alsa));
	alsa_min[d->ch] = min(alsa_min[d->ch], 
		get_remain_ringbuf(&pcmif->alsa));
	
	if (time_after(jiffies, j + HZ)) {
		DPRINTF("ch%d_%s: hw(max:%05x,min:%05x) "
			"alsa(max:%05x,min:%05x)\n", 
			d->ch, mn2ws_pcm_pcmif_get_type(pcmif), 
			(int)hw_max[d->ch], 
			(int)hw_min[d->ch], 
			(int)alsa_max[d->ch], 
			(int)alsa_min[d->ch]);
		
		j = jiffies;
		hw_max[d->ch] = 0;
		hw_min[d->ch] = -1;
		alsa_max[d->ch] = 0;
		alsa_min[d->ch] = -1;
	}
}
